from hytest import *
import os
import sys
import json
import logging
import threading
import time
from datetime import datetime
from queue import Queue

# 获取当前脚本的绝对路径
current_dir = os.path.dirname(os.path.abspath(__file__))
# 构建预定系统的绝对路径
预定系统_path = os.path.abspath(os.path.join(current_dir, '..', '..', '..'))
# 添加路径
sys.path.append(预定系统_path)
# 导入模块
try:
    from 预定系统.Base.Mqtt_Send import *
except ModuleNotFoundError as e:
    print(f"ModuleNotFoundError: {e}")
    print("尝试使用绝对路径导入")
    from 预定系统.Base.Mqtt_Send import *

# 配置日志
logging.basicConfig(
    level=logging.INFO,
    format='%(asctime)s - %(levelname)s - %(message)s'
)

# 获取当前脚本所在的目录
current_dir = os.path.dirname(os.path.abspath(__file__))
print("当前脚本所在的目录:", current_dir)
# 构建CSV文件的绝对路径
csv_file_path = os.path.join(current_dir, '../TestData/ReservationSystem_DoorScreen/MQTT心跳上报_100条.csv')


class StableMQTTClient:
    def __init__(self, broker_address, port, username, password, client_id):
        """
        初始化MQTT客户端实例

        参数:
            broker_address (str): MQTT代理服务器的地址/IP
            port (int): MQTT代理服务器的端口号
            username (str): 连接代理服务器的用户名
            password (str): 连接代理服务器的密码
            client_id (str): 客户端的唯一标识符

        功能:
            1. 保存连接参数
            2. 初始化客户端对象为None
            3. 自动尝试连接代理服务器
        """
        # 保存MQTT连接参数
        self.broker_address = broker_address  # 代理服务器地址
        self.port = port  # 端口号
        self.username = username  # 用户名
        self.password = password  # 密码
        self.client_id = client_id  # 客户端ID

        # 初始化MQTT客户端对象(将在connect方法中实例化)
        self.client = None

        # 自动尝试连接代理服务器
        self.connect()

    def connect(self):
        """
        尝试连接到MQTT broker，最多重试3次。

        每次连接失败后会等待递增的时间后重试（5秒、10秒、15秒）。
        如果所有尝试都失败，则抛出最后的异常并记录错误日志。
        连接成功时会记录成功日志并返回True。

        Returns:
            bool: 连接成功返回True，否则抛出异常。

        Raises:
            Exception: 当所有重试尝试都失败时，抛出最后一次连接尝试的异常。
        """
        max_retries = 3
        for attempt in range(max_retries):
            try:
                # 创建MQTT客户端并尝试连接
                self.client = Mqtt(self.broker_address, self.port,
                                   self.username, self.password, self.client_id)
                self.client.set_message_type("json")
                self.client.connect()
                logging.info(f"连接成功，Client ID: {self.client_id}")
                return True
            except Exception as e:
                # 最后一次尝试失败时直接抛出异常
                if attempt == max_retries - 1:
                    logging.error(f"连接失败 (尝试 {attempt + 1}/{max_retries}): {str(e)}")
                    raise

                # 非最后一次失败时等待递增时间后重试
                wait_time = (attempt + 1) * 5
                logging.warning(f"连接失败，{wait_time}秒后重试... ({attempt + 1}/{max_retries})")
                time.sleep(wait_time)

    def publish(self, topic, message):
        """
        发布消息到指定主题。

        如果发布失败，将自动尝试重新连接并重试一次发布操作。

        参数:
            topic (str): 要发布消息的主题名称
            message (str): 要发布的消息内容

        异常:
            如果重试后仍然失败，将通过logging记录错误但不会抛出异常
        """
        try:
            # 尝试发布消息
            self.client.publish(topic, message)
        except Exception as e:
            # 发布失败时记录错误并尝试重新连接后重试
            logging.error(f"发布消息失败: {str(e)}，尝试重新连接...")
            self.connect()
            self.client.publish(topic, message)  # 重试一次


# 工作线程函数
def worker(mqtt_client, config_queue, interval):
    """MQTT消息发布工作线程

    持续从配置队列中获取配置信息，构建MQTT消息并发布到指定主题。
    该线程会循环运行直到被外部中断。

    Args:
        mqtt_client: 已连接的MQTT客户端实例，用于发布消息
        config_queue: 包含配置信息的队列，每个配置项应包含topic等必要字段
        interval: 每次消息发布后的间隔时间(秒)
    """
    while True:
        # 从队列获取配置信息
        config = config_queue.get()
        try:
            # 构建并发布MQTT消息
            topic = config["topic"]
            current_time = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
            message = Mqtt.build_message(config, current_time, topic)

            mqtt_client.publish(topic, message)
            time.sleep(interval)

        except Exception as e:
            # 异常处理：记录错误日志并短暂等待
            logging.error(f"线程 {threading.current_thread().name} 发送消息失败: {e}")
            time.sleep(5)  # 出错后等待5秒
        finally:
            # 标记队列任务完成
            config_queue.task_done()


if __name__ == "__main__":
    # 读取配置文件
    configs = Mqtt.read_config_from_csv(csv_file_path)
    broker_address = "192.168.5.235"
    username = "mqtt@cmdb"
    password = "mqtt@webpassw0RD"
    port = 1883
    num_threads = 100  # 线程数量

    # 创建配置队列
    config_queue = Queue()

    # 创建稳定的MQTT客户端列表
    mqtt_clients = []
    for i, config in enumerate(configs):
        client_id = config.get("clientId", f"python_client_{i}")
        try:
            mqtt_client = StableMQTTClient(broker_address, port, username, password, client_id)
            mqtt_clients.append(mqtt_client)
        except Exception as e:
            logging.error(f"创建客户端 {client_id} 失败: {str(e)}")

    try:
        # 创建工作线程
        threads = []
        for i in range(num_threads):
            t = threading.Thread(
                target=worker,
                args=(mqtt_clients[i % len(mqtt_clients)], config_queue, 1),
                name=f"Worker-{i + 1}",
                daemon=True
            )
            t.start()
            threads.append(t)
            time.sleep(0.1)  # 避免同时启动太多线程

        # 主循环 - 持续运行
        logging.info("系统已启动，将持续运行...")
        while True:
            try:
                # 将配置放入队列
                for config in configs:
                    config_queue.put(config)

                config_queue.join()  # 等待所有任务完成
                time.sleep(1)

            except KeyboardInterrupt:
                logging.info("接收到中断信号，准备退出...")
                break
            except Exception as e:
                logging.error(f"主循环发生错误: {e}", exc_info=True)
                time.sleep(5)  # 出错后等待5秒

    except Exception as e:
        logging.error(f"发生错误: {e}", exc_info=True)
    finally:
        logging.info("正在关闭所有连接...")
        # 断开所有MQTT连接
        for client in mqtt_clients:
            try:
                client.client.disconnect()
            except:
                pass
